<?php
namespace Arkfield\WooCommerce;

use Arkfield\Misc\Utils;

if ( ! class_exists( __NAMESPACE__ . '\Woo_Commerce_Setup' ) ) {
	/**
	 * Class Woocommerce_Setup
	 * @package Arkfield\Woocommerce
	 */
	class Woo_Commerce_Setup {

		/**
		 * @var Utils
		 */
		protected $utils;

		/**
		 * @var Woo_Commerce_Settings
		 */
		protected $woo_commerce_settings;

		/**
		 * @var Woo_Commerce_Utils
		 */
		protected $woo_commerce_utils;

		/**
		 * Woo_Commerce_Setup constructor.
		 *
		 * @param Utils                 $utils
		 * @param Woo_Commerce_Settings $woo_commerce_settings
		 * @param Woo_Commerce_Utils    $woo_commerce_utils
		 */
		public function __construct(
			Utils $utils, Woo_Commerce_Settings $woo_commerce_settings, Woo_Commerce_Utils $woo_commerce_utils
		) {
			$this->utils                 = $utils;
			$this->woo_commerce_settings = $woo_commerce_settings;
			$this->woo_commerce_utils    = $woo_commerce_utils;
		}


		public function run() {
			if ( ! $this->woo_commerce_utils->is_shop_activated() ) {
				$this->remove_db_products_sections_from_page_builder();

				return;
			}

			\add_action( 'after_setup_theme', function () {
				\add_theme_support( 'woocommerce' );
			} );

			$this->remove_woo_commerce_hooks();
			$this->change_functions_priority_for_single_product_summary_action();
			$this->change_hook_for_template_loop_price();
			$this->remove_product_headings();
			$this->manage_add_to_cart_fragments();
			$this->include_out_of_stock_templates();
			$this->remove_default_stock_html();
			$this->add_config_to_page_builder();
			$this->change_hook_for_template_cross_sell_display();
			$this->woo_commerce_settings->manage_settings();
		}

		protected function remove_woo_commerce_hooks() {
			\remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper' );
			\remove_action( 'woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end' );
			\remove_action( 'woocommerce_sidebar', 'woocommerce_get_sidebar' );
			\remove_action( 'woocommerce_before_subcategory_title', 'woocommerce_subcategory_thumbnail' );
			\remove_action( 'woocommerce_before_shop_loop', 'woocommerce_result_count', 20 );
			\remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_rating', 5 );
			\remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_price' );
		}

		/**
		 * the changed hooks:
		 *
		 * woocommerce_cart_collaterals hook
		 *
		 * @hooked woocommerce_cart_totals - 10
		 *
		 * * * * * * * * * * * * * * * * * * * * * * * * * *
		 *
		 * woocommerce_after_cart hook
		 *
		 * @hooked woocommerce_cross_sell_display - 10
		 */
		protected function change_hook_for_template_cross_sell_display() {
			$this->utils->change_function_hook(
					'woocommerce_cart_collaterals',
					'woocommerce_after_cart',
					'woocommerce_cross_sell_display'
			);
		}

		/**
		 * changes the order of woocommerce_single_product_summary functions
		 *
		 * new order:
		 *
		 * @hooked woocommerce_template_single_title - 5
		 * @hooked woocommerce_template_single_rating - 9 (old 10)
		 * @hooked woocommerce_template_single_price - 10
		 * @hooked woocommerce_template_single_excerpt - 20
		 * @hooked woocommerce_template_single_add_to_cart - 30
		 * @hooked woocommerce_template_single_meta - 40
		 * @hooked woocommerce_template_single_sharing - 50
		 */
		protected function change_functions_priority_for_single_product_summary_action() {
			$this->utils->change_hook_function_priority(
				'woocommerce_single_product_summary', 'woocommerce_template_single_rating', 10, 9 );
		}

		/**
		 * the changed hooks:
		 *
		 * woocommerce_shop_loop_item_title hook
		 *
		 * @hooked woocommerce_template_loop_product_title - 10
		 * @hooked woocommerce_template_loop_price - 15
		 *
		 * * * * * * * * * * * * * * * * * * * * * * * * * *
		 *
		 * woocommerce_after_shop_loop_item_title hook
		 *
		 * @hooked woocommerce_template_loop_rating - 5
		 */
		protected function change_hook_for_template_loop_price() {
			$this->utils->change_function_hook(
				'woocommerce_after_shop_loop_item_title',
				'woocommerce_shop_loop_item_title',
				'woocommerce_template_loop_price',
				10,
				15
			);
		}

		protected function remove_product_headings() {
			\add_filter( 'woocommerce_product_additional_information_heading', function () {
				return '';
			} );

			\add_filter( 'woocommerce_product_description_heading', function () {
				return '';
			} );
		}

		protected function manage_add_to_cart_fragments() {
			\add_action( 'woocommerce_add_to_cart_fragments', function ( $fragments ) {
				$fragments['total_count'] = WC()->cart->get_cart_contents_count();

				return $fragments;
			} );
		}

		protected function include_out_of_stock_templates() {
			add_action( 'woocommerce_before_shop_loop_item_title', function () {
				\wc_get_template( 'loop/out-of-stock-flash.php' );
			} );

			add_action( 'woocommerce_before_single_product_summary', function () {
				\wc_get_template( 'single-product/out-of-stock-flash.php' );
			} );
		}

		protected function remove_default_stock_html() {
			\add_filter( 'woocommerce_stock_html', function () {
				return '';
			} );

		}

		protected function add_config_to_page_builder() {
			\add_filter( 'arkfield' . '_page_builder_section_fields', function ( $fields ) {
				$fields[0]['options']['products'] = esc_html__( 'Products', 'arkfield' );

				$fields[] = array(
					'id'          => 'nb_products',
					'type'        => 'number',
					'label'       => esc_html__( 'Number of products', 'arkfield' ),
					'description' => esc_html__( 'Select the number of products to display.', 'arkfield' ),
					'default'     => 9,
					'attributes'  => array(
						'min' => 1
					)
				);

				$fields[] = array(
					'id'          => 'products_speed',
					'type'        => 'checkbox',
					'label'       => esc_html__( 'Products speed', 'arkfield' ),
					'description' => esc_html__( 'Check if you want the products to move during scrolling', 'arkfield' ),
					'default'     => 'on',
				);

				return $fields;
			} );
		}

		protected function remove_db_products_sections_from_page_builder() {
			\add_filter( 'arkfield' . '_page_builder_db_page_sections', function ( $sections ) {
				foreach ( $sections as $index => $section ) {
					if ( 'products' === $section['content_type'] ) {
						unset( $sections[ $index ] );
					}
				}

				return $sections;
			} );
		}
	}
}
