<?php
/**
 * Functions definitions.
 *
 * @package     wpum-user-verification
 * @copyright   Copyright (c) 2016, Alessandro Tesoro
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Retrieve the selected verification method.
 *
 * @since 1.0.0
 * @return string
 */
function wpumuv_get_method() {

  $default_method = 'email';

  if( ! wpum_get_option( 'user_verification_method' ) )  {
    $default_method = wpum_get_option( 'user_verification_method' );
  }

  return $default_method;

}

/**
 * Verify if the a user has been approved or not.
 *
 * @since 1.0.0
 * @param string $user_id the id of the user.
 * @return boolean
 */
function wpumuv_is_user_approved( $user_id ) {

  $needs_verification = get_user_meta( $user_id, 'wpumuv_needs_verification', true );

  if( $needs_verification ) {

    $user_status = get_user_meta( $user_id, 'wpumuv_status', true );

    if( $user_status === 'approved' ) {
      return true;
    } else {
      return false;
    }

  }

  return true;

}

/**
 * Removes the metas previous registered needed to set the user as pending.
 *
 * @since 1.0.0
 * @return void
 */
function wpumuv_delete_verification_status( $user_id ) {

  delete_user_meta( $user_id, 'wpumuv_needs_verification' );
  delete_user_meta( $user_id, 'wpumuv_status' );
  delete_user_meta( $user_id, 'wpumuv_verification_key' );

}

/**
 * Create a verification key for the user.
 *
 * @since 1.0.0
 * @param string $user_id the ID for which we're going to set a key.
 */
function wpumuv_set_verification_key( $user_id ) {

  $verification_key = uniqid( md5( $user_id ) );

  update_user_meta( $user_id, 'wpumuv_verification_key', $verification_key );

}

/**
 * Get the verification key of a user.
 *
 * @since 1.0.0
 * @param  string $user_id the ID for which we're going to set a key.
 * @return mixed
 */
function wpumuv_get_verification_key( $user_id ) {

  return get_user_meta( $user_id, 'wpumuv_verification_key', true );

}

/**
 * Retrieve the link to verify the user.
 *
 * @since 1.0.0
 * @param  string $user_id the ID for which we're going to set a key.
 * @return string
 */
function wpumuv_get_verification_key_link( $user_id ) {

  $login_page      = wpum_get_core_page_id( 'login' );
  $confirm_url     = add_query_arg( array(
    'verify_user'      => $user_id,
    'verification_key' => wpumuv_get_verification_key( $user_id )
  ), get_permalink( $login_page ) );

  $output = '<a href="'. $confirm_url .'">'. esc_html__( 'Verify account', 'wpum-user-verification' ) .'</a>';

  return $output;

}

/**
 * Send an email to the user notifying him his registration has been rejected.
 *
 * @since 1.0.0
 * @param  string $user_id the user to notify.
 * @return void
 */
function wpumuv_send_rejection_email( $user_id ) {

  if ( wpum_email_exists( 'reject' ) ) {

		// Retrieve the email from the database.
		$rejection_email = wpum_get_email( 'reject' );
		$message         = wpautop( $rejection_email['message'] );
		$message         = wpum_do_email_tags( $message, $user_id, null );

		WPUM()->emails->__set( 'heading', esc_html__( 'Account Rejected', 'wpum-user-verification' ) );
		WPUM()->emails->send( wpum_get_user_email( $user_id ), $rejection_email['subject'], $message );

	}

}

/**
 * Send an email to the user notifying him his registration has been approved.
 *
 * @since 1.0.0
 * @param  string $user_id the user to notify.
 * @return void
 */
function wpumuv_send_approval_email( $user_id ) {

  if ( wpum_email_exists( 'approval' ) ) {

		// Retrieve the email from the database.
		$approval_email = wpum_get_email( 'approval' );
		$message        = wpautop( $approval_email['message'] );
		$message        = wpum_do_email_tags( $message, $user_id, null );

		WPUM()->emails->__set( 'heading', esc_html__( 'Account Approved', 'wpum-user-verification' ) );
		WPUM()->emails->send( wpum_get_user_email( $user_id ), $approval_email['subject'], $message );

	}

}

/**
 * Send an email to the user with a verification link.
 *
 * @since 1.0.0
 * @param  string $user_id the user to notify.
 * @return void
 */
function wpumuv_send_verification_email( $user_id ) {

  if ( wpum_email_exists( 'verification' ) ) {

		// Retrieve the email from the database.
		$verification_email = wpum_get_email( 'verification' );
		$message            = wpautop( $verification_email['message'] );
		$message            = wpum_do_email_tags( $message, $user_id, null );

		WPUM()->emails->__set( 'heading', esc_html__( 'Account Verification', 'wpum-user-verification' ) );
		WPUM()->emails->send( wpum_get_user_email( $user_id ), $verification_email['subject'], $message );

	}

}

/**
 * Retrieves a list of users that haven't activated their profiles within 30 days.
 *
 * @since 1.0.0
 * @return array
 */
function wpumuv_get_expired_users() {

  $user_query = new WP_User_Query( array(
    'role__not_in' => array( 'Administrator' ),
    'number'       => 9999,
    'fields'       => 'ID',
    'meta_key'     => 'wpumuv_needs_verification',
    'meta_value'   => true,
    'date_query'   => array(
      array( 'before' => '29 days ago midnight' ),
      array( 'after'  => '30 days ago midnight', 'inclusive' => true ),
    )
  ) );

  $users = $user_query->get_results();

  return $users;

}
