<?php
namespace Arkfield\Site\Blog;

if ( ! class_exists( __NAMESPACE__ . '\\Blog_DAL' ) ) {
	/**
	 * Class Blog_DAL
	 * @package Arkfield\Site\Blog
	 */
	class Blog_DAL {
		/**
		 * Retrieve the name of all post categories
		 *
		 * @param int $post_id the post id.
		 *
		 * @return array
		 */
		public function get_post_categories_name( $post_id ) {
			return \wp_get_object_terms( $post_id, 'category', array( 'fields' => 'names' ) );
		}

		/**
		 * @param int   $user_id
		 * @param array $user_contact_methods
		 *
		 * @return array
		 */
		public function get_author_social_networks_data( $user_id, array $user_contact_methods ) {
			$social_links = array();

			array_map( function ( $item ) use ( $user_id, &$social_links ) {
				$data = \get_the_author_meta( $item, $user_id );

				if ( ! empty( $data ) ) {
					$social_links[ $item ] = $data;
				}
			}, $user_contact_methods );

			return $social_links;
		}

		/**
		 * gets the related posts of the current post
		 *
		 * first it tries to find related posts by post tags
		 * if it cannot find sufficient posts by tags it gets by post categories
		 *
		 * @param int $post_id
		 * @param int $nb_posts
		 *
		 * @return \WP_Post[]
		 */
		public function get_related_posts( $post_id, $nb_posts = 5 ) {
			$related_posts = array();

			$tags = \wp_get_post_tags( $post_id );

			if ( $tags ) {
				$tag_ids = array_map( function ( $tag ) {
					return $tag->term_id;
				}, $tags );

				$related_posts = get_posts( array(
					'posts_per_page' => $nb_posts,
					'exclude'        => array( $post_id ),
					'tax_query'      => array(
						array(
							'taxonomy' => 'post_tag',
							'field'    => 'term_id',
							'terms'    => $tag_ids,
						),
					)
				) );

				if ( count( $related_posts ) === $nb_posts ) {
					return $related_posts;
				}
			}

			$category_ids = \wp_get_post_categories( $post_id );

			if ( ! $category_ids ) {
				return $related_posts;
			}

			$related_tag_post_ids = array( $post_id );
			foreach ( $related_posts as $related_post ) {
				$related_tag_post_ids[] = $related_post->ID;
			}

			return array_merge(
				$related_posts,
				\get_posts( array(
					'posts_per_page' => $nb_posts - count( $related_posts ),
					'exclude'        => $related_tag_post_ids,
					'category'       => implode( ',', $category_ids )
				) )
			);
		}

		/**
		 * Retrieve list of posts matching criteria and loops through them applying the specified callback
		 *
		 * @param array    $args
		 * @param \closure $callback
		 */
		public function get_posts_and_process_loop( array $args, \closure $callback ) {
			$query = new \WP_Query( array_merge( array(
				'ignore_sticky_posts' => true,
				'no_found_rows'       => true,
				'orderby'             => 'date',
				'order'               => 'DESC',
			), $args ) );

			if ( $query->have_posts() ) {
				while ( $query->have_posts() ) {
					$query->the_post();
					// dirty hack in order to display only the content before the more tag, for wp < 4.1
					$GLOBALS['more'] = 0;
					$callback();
				}
			}

			\wp_reset_postdata();
		}
	}
}