<?php
namespace Arkfield\Project;

use Arkfield\Misc\Utils;

if ( ! class_exists( __NAMESPACE__ . '\Project_Utils' ) ) {
	/**
	 * Class Project_Utils
	 * @package Arkfield\Project
	 */
	class Project_Utils {
		/**
		 * @var Project_Config
		 */
		protected $project_config;

		/**
		 * @var Project_DAL
		 */
		protected $project_dal;

		/**
		 * @var Utils
		 */
		protected $utils;

		/**
		 * Project_Utils constructor.
		 *
		 * @param Project_Config $project_config
		 * @param Project_DAL    $project_dal
		 * @param Utils          $utils
		 */
		public function __construct( Project_Config $project_config, Project_DAL $project_dal, Utils $utils ) {
			$this->project_config = $project_config;
			$this->project_dal    = $project_dal;
			$this->utils          = $utils;
		}

		/**
		 * @param string $category_id
		 * @param string $category_slug
		 *
		 * @return string
		 */
		public function get_project_category_filter_option( $category_id, $category_slug ) {
			return $category_slug . '-' . $category_id;
		}

		/**
		 * @param \WP_Post $project
		 *
		 * @return string
		 */
		public function get_project_categories_filter_options( \WP_Post $project ) {
			$categories = $this->project_dal->get_project_categories_grouped_by_id( $project );
			$options    = array();
			foreach ( $categories as $id => $category ) {
				$options[] = $this->get_project_category_filter_option( $id, $category->slug );
			}

			return implode( ' ', $options );
		}

		/**
		 * @param int $project_index
		 *
		 * @return int
		 */
		public function get_classic_projects_animation_delay( $project_index ) {
			$animation_delay_config = $this->project_config->get_classic_projects_animation_delay_config();

			return $animation_delay_config[ $project_index % count( $animation_delay_config ) ];
		}

		/**
		 * @param int $project_index
		 *
		 * @return int
		 */
		public function get_full_grid_projects_animation_delay( $project_index ) {
			$animation_delay_config = $this->project_config->get_full_grid_projects_animation_delay_config();

			return $animation_delay_config[ $project_index % count( $animation_delay_config ) ];
		}

		/**
		 * @param string $project_id
		 * @param string $project_title
		 * @param array  $project_meta
		 * @param string $size
		 * @param array  $attributes
		 *
		 * @return string
		 */
		public function get_project_featured_image_html(
			$project_id,
			$project_title,
			$project_meta,
			$size,
			$attributes = array()
		) {
			$attributes = array_merge( array( 'title' => $project_title ), $attributes );

			$post_thumbnail = $this->project_dal->get_project_thumbnail( $project_id, $size, $attributes );

			if ( ! empty( $post_thumbnail ) ) {
				return $post_thumbnail;
			}

			$gallery_ids = $this->get_project_gallery_ids( $project_meta );

			if ( empty( $gallery_ids ) ) {
				return sprintf( '<img src="%s" alt="%s" %s />',
					sprintf( $this->project_config->get_project_no_feature_image_url_pattern(), $size ),
					$project_title, $this->utils->html_attributes_to_string( $attributes ) );
			}

			$attributes['class'] = ( isset( $attributes['class'] ) ? $attributes['class'] . ' ' : '' ) . 'wp-post-image';

			return $this->project_dal->get_project_image( array_shift( $gallery_ids ), $size, $attributes );
		}

		/**
		 * @param array $project_meta
		 *
		 * @return array
		 */
		public function get_project_gallery_ids( array $project_meta ) {
			return isset( $project_meta['gallery'] ) && ! empty( $project_meta['gallery'] )
				? explode( ',', $project_meta['gallery'] ) : array();
		}

		/**
		 * @param string $title
		 * @param string $meta_title
		 *
		 * @return mixed|void
		 */
		public function get_project_title( $title, $meta_title ) {
			return empty ( $meta_title ) ? $title : $this->get_filtered_title( $meta_title );
		}

		/**
		 * @param $title
		 *
		 * @return string
		 */
		public function get_filtered_title( $title ) {
			return \apply_filters( 'the_title', $title );
		}
	}
}