<?php
namespace Arkfield\Admin\ThemeCustomizer;

if ( ! class_exists( __NAMESPACE__ . '\\Theme_Options_Utils' ) ) {
	/**
	 * Class Theme_Options_Utils
	 * @package Arkfield\Admin\ThemeCustomizer
	 */
	class Theme_Options_Utils {

		/**
		 * @var Theme_Options_Config
		 */
		protected $theme_options_config;

		/**
		 * @var Theme_Options_Css_Filter
		 */
		protected $theme_options_css_filter;

		/**
		 * @var Google_Fonts_Options
		 */
		protected $google_fonts_options;

		/**
		 * @var array
		 */
		protected $controls;

		/**
		 * Theme_Options_Utils constructor.
		 *
		 * @param Theme_Options_Config     $theme_options_config
		 * @param Theme_Options_Css_Filter $theme_options_css_filter
		 * @param Google_Fonts_Options     $google_fonts_options
		 */
		public function __construct(
			Theme_Options_Config $theme_options_config,
			Theme_Options_Css_Filter $theme_options_css_filter,
			Google_Fonts_Options $google_fonts_options
		) {
			$this->theme_options_config     = $theme_options_config;
			$this->theme_options_css_filter = $theme_options_css_filter;
			$this->google_fonts_options     = $google_fonts_options;
			$this->controls                 = $this->theme_options_config->get_controls();
		}

		/**
		 * @param string $id
		 *
		 * @return string
		 */
		public function get_custom_option( $id ) {
			$default = isset( $this->controls[ $id ]['default'] ) ? $this->controls[ $id ]['default'] : false;

			return \get_theme_mod( $this->build_setting_id( $id ), $default );
		}

		/**
		 * @return array
		 */
		public function get_color_options() {
			$color_options = array();

			foreach ( $this->theme_options_config->get_colors_section_controls() as $control ) {
				$color_options[ $control['id'] ] = $this->get_custom_option( $control['id'] );
			}

			return $color_options;
		}

		/**
		 * @param string $setting_id
		 *
		 * @return string
		 */
		public function build_setting_id( $setting_id ) {
			return 'arkfield' . '_setting_' . $setting_id;
		}

		/**
		 * @param string $id
		 *
		 * @return string
		 */
		public function build_panel_id( $id ) {
			return 'arkfield' . '_panel_' . $id;
		}

		/**
		 * @param string $id
		 *
		 * @return string
		 */
		public function build_section_id( $id ) {
			return 'arkfield' . '_section_' . $id;
		}

		/**
		 * @return array
		 */
		public function get_customized_js() {
			$style_config = array();
			$text_config  = array();
			$css_filters  = array();

			foreach ( $this->controls as $control_id => $control ) {
				$setting_id = $this->build_setting_id( $control_id );

				if ( isset( $control['css'] ) ) {
					$style_config[ $setting_id ] = $control['css'];
				}

				if ( isset( $control['css_items'] ) ) {
					$text_config[ $setting_id ] = $control['css_items'];
				}

				if ( isset( $control['css_filter'] ) ) {
					$css_filters[ $setting_id ] = $control['css_filter'];
				}
			}

			return array(
				'style'      => $style_config,
				'text'       => $text_config,
				'cssFilters' => $css_filters,
			);
		}

		/**
		 * @return string
		 */
		public function get_customized_css() {
			$style_data = array();

			foreach ( $this->controls as $control_id => $control ) {
				if ( ! isset( $control['css'] ) ) {
					continue;
				}

				$css_filter_method        = isset( $control['css_filter'] ) ? $control['css_filter'] : null;
				$css_filter_method_exists = null !== $css_filter_method && method_exists( $this->theme_options_css_filter, $css_filter_method );

				foreach ( $control['css'] as $css_property => $css_elements ) {
					$option = $this->get_custom_option( $control_id );

					if ( $css_filter_method_exists ) {
						$option = $this->theme_options_css_filter->$css_filter_method( $option );
					}

					$style_data[] = $this->get_css_string( $css_elements, $css_property, $option );
				}
			}

			return implode( PHP_EOL, $style_data );
		}

		/**
		 * @return array
		 */
		public function get_logo_size() {
			$logo_control_id = 'logo_dimension';
			$dimension       = $this->get_custom_option( $logo_control_id );

			if ( false === $dimension ) {
				return array(
					'width'      => 0,
					'height'     => 0,
					'max-width'  => 0,
					'max-height' => 0,
				);
			}

			$logo_config = $this->controls[ $logo_control_id ]['input_attrs'];

			$compute_real_logo_dimension = function ( $dim_type ) use ( $logo_control_id, $dimension, $logo_config ) {
				$max_dim = $logo_config[ 'data-max-' . $dim_type ];
				$min_dim = $logo_config[ 'data-min-' . $dim_type ];

				return ( $max_dim - $min_dim ) * $dimension / $logo_config['max'] + $min_dim;
			};

			return array(
				'width'      => $compute_real_logo_dimension( 'width' ),
				'height'     => $compute_real_logo_dimension( 'height' ),
				'max-width'  => $logo_config['data-max-width'],
				'max-height' => $logo_config['data-max-height'],
			);
		}

		public function enqueue_fonts() {
			$this->google_fonts_options->enqueue_google_fonts( array(
				$this->get_custom_option( 'heading_font_family' ),
				$this->get_custom_option( 'text_font_family' )
			) );
		}

		/**
		 * @param array  $css_elements
		 * @param string $css_property
		 * @param string $css_value
		 *
		 * @return string
		 */
		protected function get_css_string( array $css_elements, $css_property, $css_value ) {
			return sprintf(
				'%s {%s: %s}',
				implode( ', ', $css_elements ),
				$css_property,
				$css_value );
		}
	}
}