<?php
namespace Arkfield\Admin;

if ( ! class_exists( __NAMESPACE__ . '\Pikart_Gallery' ) ) {
	/**
	 * Class Pikart_Gallery
	 * @package Arkfield\Admin
	 */
	class Pikart_Gallery {
		const NONCE_ACTION = 'update_pikart_gallery';
		const NONCE_NAME = 'pikart_gallery_nonce';

		public function setup() {
			$this->register_gallery_update_ajax();
			$this->enqueue_scripts();
		}

		/**
		 * @param string $id        id of the gallery
		 * @param array  $image_ids list of image ids, ex: [10, 2, 34]
		 *
		 * @return string
		 */
		public function generate_gallery_html( $id, array $image_ids ) {

			$html =
				'<div class="pikart-gallery-container">
					<a href="#" class="pikart-gallery-open">%s</a>
					<a href="#" class="pikart-gallery-reset">%s</a>
					<input type="hidden" name="%s" id="%s" value="%s"/>
					<ul class="pikart-gallery-images">%s</ul>
				</div>';

			$html .= wp_nonce_field( self::NONCE_ACTION, self::NONCE_NAME, true, false );

			return sprintf(
				$html,
				esc_html__( 'Open', 'arkfield' ),
				esc_html__( 'Reset', 'arkfield' ),
				$id,
				$id,
				implode( ',', $image_ids ),
				$this->generate_image_list_html( $image_ids )
			);
		}

		protected function register_gallery_update_ajax() {
			$generate_image_list_html_callback = $this->generate_image_list_html_callback();

			\add_action( 'wp_ajax_pikart_gallery_update_ajax', function () use ( $generate_image_list_html_callback ) {
				$nonce_value = filter_input( INPUT_POST, Pikart_Gallery::NONCE_NAME, FILTER_SANITIZE_STRING );
				$image_ids   = filter_input( INPUT_POST, 'galleryIds', FILTER_UNSAFE_RAW, FILTER_REQUIRE_ARRAY );

				if ( ! $nonce_value || ! \wp_verify_nonce( $nonce_value, Pikart_Gallery::NONCE_ACTION ) || ! $image_ids ) {
					\wp_send_json_error();
				}

				\wp_send_json_success( $generate_image_list_html_callback( $image_ids ) );
			} );
		}

		protected function enqueue_scripts() {
			\add_action( 'admin_enqueue_scripts', function ( $hook ) {
				if ( 'post-new.php' === $hook || 'post.php' === $hook ) {
					\wp_enqueue_media();
					\wp_enqueue_script( 'pikart-gallery-script', ARKFIELD_THEME_JS_URI . '/admin/pikart-gallery.js',
						array( 'jquery' ), ARKFIELD_THEME_VERSION, true
					);
				}
			} );
		}

		/**
		 * @param array $image_ids
		 *
		 * @return string
		 */
		protected function generate_image_list_html( array $image_ids ) {
			$generate_image_list_html_callback = $this->generate_image_list_html_callback();
			return $generate_image_list_html_callback($image_ids);
		}

		/**
		 * @param array $image_ids
		 *
		 * @return string
		 */
		protected function generate_image_list_html_callback() {
			return function (array $image_ids) {
				return implode( '', array_map( function ( $img_id ) {
					return sprintf( '<li>%s</li>', wp_get_attachment_image( $img_id, 'thumbnail' ) );
				}, $image_ids ) );
			};
		}
	}
}