<?php
/**
 * Handles hooks.
 *
 * @package     wpum-user-verification
 * @copyright   Copyright (c) 2016, Alessandro Tesoro
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Add the required user metas upon registration.
 * This is needed to trigger user verification.
 *
 * @since 1.0.0
 * @param string $user_id the user to work with.
 */
function wpumuv_add_verification_meta( $user_id, $values ) {

  update_user_meta( $user_id, 'wpumuv_needs_verification', true );
  update_user_meta( $user_id, 'wpumuv_status', 'pending' );
  wpumuv_set_verification_key( $user_id );

}
add_action( 'wpum/form/register/success', 'wpumuv_add_verification_meta', 10, 2 );

/**
 * Show a notice telling the admin something is wrong with his settings.
 *
 * @since 1.0.0
 * @return void
 */
function wpumuv_compatibility_notice() {

  if( wpum_get_option('login_after_registration') ) : ?>
  <div class="notice notice-error">
      <p><?php esc_html_e( 'When user verification is enabled, you must disable the automatic login option.', 'wpum-user-verification' ); ?></p>
      <a href="<?php echo esc_url( admin_url( 'users.php?page=wpum-settings&tab=registration#wpum_settings[login_after_registration]' ) ); ?>" class="button"><?php esc_html_e( 'Disable option', 'wpum-user-verification' ); ?></a>
      <br/><br/>
  </div>
  <?php endif;

}
add_action( 'admin_notices', 'wpumuv_compatibility_notice' );

/**
 * New email tag.
 *
 * @param string $user_id the user to check.
 */
function wpumuv_add_email_verification_link_tag( $user_id ) {
	 wpum_add_email_tag( 'verification_link', 'Use this tag to display the user account verification link', 'wpumuv_get_verification_key_link' );
}
add_action( 'wpum_add_email_tags', 'wpumuv_add_email_verification_link_tag' );

/**
 * Show a message telling the user the status of the verification.
 *
 * @since 1.0.0
 * @return void
 */
function wpumuv_show_verification_message( $args ) {

  if( isset( $_GET['verification'] ) && $_GET['verification'] == '1' ) {

    $args = array(
		  'id'   => 'wpum-verification-success',
		  'type' => 'success',
		  'text' => esc_html__( 'Thank you for verifying your account. You can now login.', 'wpum-user-verification' )
		);

    $warning = wpum_message( apply_filters( 'wpumuv_success_message', $args ), true );

  } elseif( isset( $_GET['verification'] ) && $_GET['verification'] == 'failed' ) {

    $args = array(
		  'id'   => 'wpum-verification-failed',
		  'type' => 'error',
		  'text' => esc_html__( 'Could not verify this account. Please try again.', 'wpum-user-verification' )
		);

    $warning = wpum_message( apply_filters( 'wpumuv_failed_message', $args ), true );

  }

}
add_action( 'wpum_before_login_form', 'wpumuv_show_verification_message', 10, 1 );

/**
 * Modify the message upon successful registration.
 *
 * @since 1.0.0
 * @param  string $text original message.
 * @return string       new message.
 */
function wpumuv_registration_success_message( $text ) {

  $method = wpumuv_get_method();

  if( $method == 'email' ) {
    $text = esc_html__( 'Registration complete. We have sent you an email with a verification link.', 'wpum-user-verification' );
  } else if( $method == 'manual' ) {
    $text = esc_html__( 'Registration complete. An administrator will review your registration. You\'ll receive an email once your account has been approved.', 'wpum-user-verification' );
  }

  return $text;

}
add_filter( 'wpum/form/register/success/message', 'wpumuv_registration_success_message', 10, 1 );

/**
 * Deletes non verified users after a set amount of time.
 *
 * @since 1.0.0
 * @return void
 */
function wpumuv_check_non_verified_users() {

  if( ! wpum_get_option( 'user_verification_removal' ) )
    return;

	$users = wpumuv_get_expired_users();

	if( ! empty( $users ) && is_array( $users ) ) {

		require_once( ABSPATH . 'wp-admin/includes/user.php' );

		foreach ( $users as $user_id ) {
			wp_delete_user( $user_id );
		}

	}

}
add_action( 'wpumuv_check_non_verified_users', 'wpumuv_check_non_verified_users' );

/**
 * Display a message within the plugin's name telling the user to add the license.
 *
 * @since 1.0.0
 * @param  string $plugin_name name of the plugin
 * @return void
 */
function wpumuv_license_message( $plugin_name ) {

  if( get_option( 'wpum_user_verification_license_active' ) !== 'valid' ) {

  $register_link = sprintf( '<a href="%s" target="_blank">' . esc_html_x( 'Register', 'used within the license activation message', 'wpum-user-verification' ) . '</a>' , '#' );
  $purchase_link = sprintf( '<a href="%s" target="_blank">' . esc_html__( 'Purchase one now.', 'wpum-user-verification' ) . '</a>', '#' );

  $message = $register_link . ' ' . sprintf( esc_html__( 'your copy of the User Verification addon to receive access to automatic upgrades and support. Need a license key?', 'wpum-user-verification' ) ) . ' ' . $purchase_link ;

  echo '</tr><tr class="plugin-update-tr"><td colspan="3" class="plugin-update"><div class="update-message">'.$message.'</div></td>';

  }

}
add_action( 'after_plugin_row_'.WPUMUV_SLUG, 'wpumuv_license_message' );
