<?php
namespace Arkfield\PageBuilder;

use Arkfield\Admin\MetaBoxes\Meta_Box_Utils;

if ( ! class_exists( __NAMESPACE__ . '\Page_Builder_Manager' ) ) {
	/**
	 * Class Page_Builder_Manager
	 * @package Arkfield\PageBuilder
	 */
	class Page_Builder_Manager {
		/**
		 * @var Page_Builder_Config
		 */
		protected $page_builder_config;

		/**
		 * @var Page_Builder_DAL
		 */
		protected $page_builder_dal;

		/**
		 * @var Page_Builder_Meta_Box
		 */
		protected $page_builder_meta_box;

		/**
		 * @var Meta_Box_Utils
		 */
		protected $meta_box_utils;

		/**
		 * Page_Builder_Manager constructor.
		 *
		 * @param Page_Builder_Config   $page_builder_config
		 * @param Page_Builder_DAL      $page_builder_dal
		 * @param Page_Builder_Meta_Box $page_builder_meta_box
		 * @param Meta_Box_Utils        $meta_box_utils
		 */
		public function __construct(
			Page_Builder_Config $page_builder_config,
			Page_Builder_DAL $page_builder_dal,
			Page_Builder_Meta_Box $page_builder_meta_box,
			Meta_Box_Utils $meta_box_utils
		) {
			$this->page_builder_config   = $page_builder_config;
			$this->page_builder_dal      = $page_builder_dal;
			$this->page_builder_meta_box = $page_builder_meta_box;
			$this->meta_box_utils        = $meta_box_utils;
		}


		public function setup() {
			$this->register_meta_boxes_generation();
			$this->register_meta_box_update();
			$this->register_meta_box_ajax_generation();
			$this->enqueue_page_builder_script();
			$this->generate_add_section();
		}

		protected function register_meta_box_update() {
			$page_builder_dal = $this->page_builder_dal;
			\add_action( 'save_post_' . Page_Builder_Meta_Box::POST_TYPE, function ( $page_id ) use ( $page_builder_dal ) {
				$page_builder_dal->update_meta_box_data( $page_id );
			} );
		}

		protected function register_meta_boxes_generation() {
			$page_builder_meta_box = $this->page_builder_meta_box;
			\add_action( 'add_meta_boxes_' . Page_Builder_Meta_Box::POST_TYPE, function ( $page ) use ( $page_builder_meta_box ) {
				$page_builder_meta_box->generate_meta_boxes( $page->ID );
			} );
		}

		protected function register_meta_box_ajax_generation() {
			$page_builder_meta_box = $this->page_builder_meta_box;
			$meta_box_utils        = $this->meta_box_utils;
			\add_action( 'wp_ajax_page_builder_add_meta_box', function () use ( $page_builder_meta_box, $meta_box_utils ) {
				$meta_box_index = filter_input( INPUT_POST, 'metaBoxNewIndex', FILTER_SANITIZE_NUMBER_INT );

				if ( null === $meta_box_index || false === $meta_box_index ) {
					wp_send_json_error();
				}

				$page_builder_meta_box->generate_meta_box_by_index( $meta_box_index );

				wp_send_json_success(
					$meta_box_utils->capture_meta_boxes_content( Page_Builder_Meta_Box::POST_TYPE, 'normal' ) );
			} );
		}

		protected function enqueue_page_builder_script() {
			\add_action( 'admin_enqueue_scripts', function ( $hook ) {
				$current_screen = get_current_screen();

				if ( Page_Builder_Meta_Box::POST_TYPE === $current_screen->post_type
				     && ( 'post-new.php' === $hook || 'post.php' === $hook )
				) {
					wp_enqueue_script( 'arkfield' . '-admin-page-builder', ARKFIELD_THEME_JS_URI . '/admin/page-builder.js',
						array( 'jquery' ), ARKFIELD_THEME_VERSION, true
					);
				}
			} );
		}

		protected function generate_add_section() {
			$page_builder_config = $this->page_builder_config;
			\add_action( 'edit_form_after_editor', function ( \WP_Post $post ) use ( $page_builder_config ) {
				if ( Page_Builder_Meta_Box::POST_TYPE === $post->post_type ) {
					echo sprintf(
						'<div id="page_builder_add_section" class="postbox"><h3><span>%s</span></h3></div>',
						$page_builder_config->get_add_section_label()
					);
				}
			} );
		}
	}
}