<?php
namespace Arkfield\PageBuilder;

use Arkfield\Admin\MetaBoxes\Meta_Box_Sanitizer;

if ( ! class_exists( __NAMESPACE__ . '\Page_Builder_DAL' ) ) {
	/**
	 * Class Page_Builder_DAL
	 * @package Arkfield\PageBuilder
	 */
	class Page_Builder_DAL {
		const PAGE_BUILDER_DB_PREFIX = 'pikart_';

		/**
		 * @var Page_Builder_Config
		 */
		protected $page_builder_config;

		/**
		 * @var Meta_Box_Sanitizer
		 */
		protected $meta_box_sanitizer;

		/**
		 * Page_Builder_DAL constructor.
		 *
		 * @param Page_Builder_Config $page_builder_config
		 * @param Meta_Box_Sanitizer  $meta_box_sanitizer
		 */
		public function __construct( Page_Builder_Config $page_builder_config, Meta_Box_Sanitizer $meta_box_sanitizer ) {
			$this->page_builder_config = $page_builder_config;
			$this->meta_box_sanitizer  = $meta_box_sanitizer;
		}


		/**
		 * @param int $page_id
		 *
		 * @return array
		 */
		public function get_meta_box_data( $page_id ) {
			return apply_filters(
				'arkfield' . '_page_builder_db_page_sections',
				unserialize( base64_decode( \get_post_meta( $page_id, $this->get_storage_meta_key(), true ) ) ),
				$page_id
			);
		}

		/**
		 * @param int $page_id
		 */
		public function update_meta_box_data( $page_id ) {
			$page_builder_sections = filter_input(
				INPUT_POST, Page_Builder_Config::PAGE_BUILDER_SECTIONS_META_KEY, FILTER_UNSAFE_RAW, FILTER_REQUIRE_ARRAY );

			if ( $this->allow_meta_box_data_update( $page_builder_sections, $page_id ) ) {
				\update_post_meta(
					$page_id,
					$this->get_storage_meta_key(),
					base64_encode( serialize( $this->sanitize_page_builder_sections_data( $page_builder_sections ) ) )
				);
			}
		}

		/**
		 * @param array $page_builder_sections
		 * @param int   $page_id
		 *
		 * @return bool
		 */
		protected function allow_meta_box_data_update( &$page_builder_sections, $page_id ) {
			return $page_builder_sections && $this->is_valid_nonce( $page_builder_sections )
			       && ! \wp_is_post_autosave( $page_id ) && \current_user_can( 'edit_post', $page_id );
		}

		/**
		 * @param array $page_builder_sections
		 *
		 * @return array
		 */
		protected function sanitize_page_builder_sections_data( array $page_builder_sections ) {
			$fields_with_type = $this->get_meta_box_section_fields_with_type();

			foreach ( $page_builder_sections as &$section ) {
				foreach ( $section as $field => &$value ) {
					if ( ! isset( $fields_with_type[ $field ] ) ) {
						$fields_with_type[ $field ] = Meta_Box_Sanitizer::DEFAULT_SANITIZE_TYPE;
					}
					$value = $this->meta_box_sanitizer->sanitize( $value, $fields_with_type[ $field ] );
				}
			}

			return $page_builder_sections;
		}

		/**
		 * @param array $page_builder_sections
		 *
		 * @return bool
		 */
		protected function is_valid_nonce( array &$page_builder_sections ) {
			$meta_boxes_page_section_config = $this->page_builder_config->get_meta_boxes_page_section_config();
			$nonce                          = $meta_boxes_page_section_config['nonce'];
			$is_valid_nonce                 = true;

			foreach ( $page_builder_sections as &$section ) {
				if ( isset( $section[ $nonce['name'] ] ) ) {
					$nonce_value    = $section[ $nonce['name'] ];
					$is_valid_nonce = $is_valid_nonce && $nonce_value && \wp_verify_nonce( $nonce_value, $nonce['action'] );
					unset( $section[ $nonce['name'] ] );
				} else {
					$is_valid_nonce = false;
					break;
				}
			}

			return $is_valid_nonce;
		}

		/**
		 * @return string
		 */
		protected function get_storage_meta_key() {
			return self::PAGE_BUILDER_DB_PREFIX . Page_Builder_Config::PAGE_BUILDER_SECTIONS_META_KEY;
		}

		/**
		 * Get the meta_boxes fields with their types
		 * Ex: ['field_id_1' => 'type1']
		 *
		 * @return array
		 */
		protected function get_meta_box_section_fields_with_type() {
			$meta_boxes_page_section_config = $this->page_builder_config->get_meta_boxes_page_section_config();
			$fields_config                  = $meta_boxes_page_section_config['fields'];
			$fields                         = array();

			foreach ( $fields_config as $field_config ) {
				$fields[ $field_config['id'] ] = $field_config['type'];
			}

			return $fields;
		}
	}
}