<?php
/**
 * Pending users table.
 *
 * @package     wpum-user-verification
 * @copyright   Copyright (c) 2016, Alessandro Tesoro
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.1.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

class WPUMUV_Pending_Users_Table extends WP_List_Table {

  /**
   * Prepare the items for the table to process.
   *
   * @since 1.0.0
   * @return Void
   */
  public function prepare_items() {

    $columns  = $this->get_columns();
    $hidden   = $this->get_hidden_columns();
    $sortable = $this->get_sortable_columns();

    $this->process_bulk_action();

    $this->_column_headers = array( $columns, $hidden, $sortable );
    $this->items = $this->table_data();

  }

  /**
   * Message to be displayed when there are no items.
   *
   * @since 1.0.0
   * @access public
   */
  public function no_items() {
    esc_html_e( 'No users pending approval.', 'wpum-user-verification' );
  }

  /**
   * Override the parent columns method. Defines the columns to use in the listing table
   *
   * @since 1.0.0
   * @return Array
   */
  public function get_columns() {

    $columns = array(
      'cb'         => '<input type="checkbox" />',
      'username'   => esc_html__( 'Username', 'wpum-user-verification' ),
      'name'       => esc_html__( 'Name', 'wpum-user-verification' ),
      'email'      => esc_html__( 'Email', 'wpum-user-verification' ),
      'registered' => esc_html__( 'Registered', 'wpum-user-verification' ),
    );

    return $columns;

  }

  /**
   * Define which columns are hidden.
   *
   * @since 1.0.0
   * @return Array
   */
  public function get_hidden_columns() {
    return array();
  }

  /**
   * Define the sortable columns.
   *
   * @since 1.0.0
   * @return Array
   */
  public function get_sortable_columns() {
    return null;
  }

  /**
   * Returns an associative array containing the bulk action.
   *
   * @since 1.0.0
   * @return array
   */
  public function get_bulk_actions() {

    $actions = array(
      'bulk-approve' => esc_html__( 'Approve', 'wpum-user-verification' ),
      'bulk-reject'  => esc_html__( 'Reject', 'wpum-user-verification' )
    );

    return $actions;

  }

  /**
   * Retrieve data for the table.
   *
   * @since 1.0.0
   * @return array
   */
  private function table_data() {

    global $wpumuv_signups_manager;

    $data = $wpumuv_signups_manager->get_pending_users();

    return $data;

  }

  /**
   * Define what data to show on each column of the table.
   *
   * @param  Array $item        Data
   * @param  String $column_name - Current column name
   *
   * @return Mixed
   */
  public function column_default( $item, $column_name ) {

    global $wpumuv_signups_manager;

    $actions = array(
      'approve_user' => '<a href="'. $wpumuv_signups_manager->get_user_approve_link( $item->ID ) .'">'. esc_html__( 'Approve', 'wpum-user-verification' ) .'</a>',
      'trash'        => '<a href="'. $wpumuv_signups_manager->get_user_reject_link( $item->ID ) .'" class="delete">'. esc_html__( 'Reject', 'wpum-user-verification' ) .'</a>',
      'resend'       => '<a href="'. $wpumuv_signups_manager->get_user_resend_link( $item->ID ) .'">'. esc_html__( 'Resend verification email', 'wpum-user-verification' ) .'</a>'
    );

    if( wpumuv_get_method() == 'manual' ) {
      unset( $actions['resend'] );
    }

    switch( $column_name ) {
      case 'cb':
        return $this->column_cb( $item );
      break;
      case 'username':
        return '<a href="'. add_query_arg( array( 'user_id' => $item->ID ), admin_url( 'user-edit.php' ) ) .'">'. esc_html( stripslashes( $item->user_login ) ) . '</a>' . $this->row_actions( $actions );
      break;
      case 'name':
        return esc_html( wpum_get_user_fname( $item->ID ) );
      break;
      case 'email':
        return esc_html( $item->user_email );
      break;
      case 'registered':
        return date_i18n( get_option( 'date_format' ), strtotime( $item->user_registered ) );
      break;
      default:
        return null;
    }

  }

  /**
   * Get a list of CSS classes for the list table table tag.
   *
   * @access protected
   * @return array List of CSS classes for the table tag.
   */
  protected function get_table_classes() {
    return array( 'widefat', 'fixed', $this->_args['plural'] );
  }

  /**
   * Generates content for a single row of the table
   *
   * @access public
   * @param object $item The current item
   */
  public function single_row( $item ) {

    static $row_class = '';

    $row_class = ( $row_class == '' ? ' class="alternate"' : '' );

    echo '<tr' . $row_class . '>';
    $this->single_row_columns( $item );
    echo '</tr>';

  }

  /**
   * Retrieve the checkbox for bulk actions.
   *
   * @since 1.0.0
   * @param  object $item the item to work with.
   * @return string
   */
  public function column_cb( $item ) {

    return sprintf(
      '<input type="checkbox" name="bulk_user[]" value="%s" />', $item->ID
    );

  }

  /**
   * Process bulk actions once submitted.
   *
   * @since 1.0.0
   * @return void
   */
  public function process_bulk_action() {

    if ( isset( $_POST['_wpnonce'] ) && ! empty( $_POST['_wpnonce'] ) ) {

      $nonce  = filter_input( INPUT_POST, '_wpnonce', FILTER_SANITIZE_STRING );
      $action = 'bulk-' . $this->_args['plural'];

      if ( ! wp_verify_nonce( $nonce, $action ) )
        wp_die( 'Nope! Security check failed!' );

      $action = $this->current_action();

      if( $action == 'bulk-approve' ) {

        if( isset( $_POST['bulk_user'] ) && ! empty( $_POST['bulk_user'] ) && is_array( $_POST['bulk_user'] ) ) {

          $users = $_POST['bulk_user'];
          $users = array_map( 'absint' , $users );

          foreach ( $users as $user_id ) {

            wpumuv_delete_verification_status( $user_id );
            wpumuv_send_approval_email( $user_id );

          }

        }

      } elseif( $action == 'bulk-reject' ) {

        if( isset( $_POST['bulk_user'] ) && ! empty( $_POST['bulk_user'] ) && is_array( $_POST['bulk_user'] ) ) {

          $users = $_POST['bulk_user'];
          $users = array_map( 'absint' , $users );

          foreach ( $users as $user_id ) {

            global $wpumuv_signups_manager;
            $wpumuv_signups_manager->delete_user( $user_id );

          }

        }

      }

    }

  }

}
